document.addEventListener('DOMContentLoaded', () => {
    const settingsBtn = document.getElementById('settings-btn');
    const settingsPanel = document.getElementById('settings-panel');
    const saveKeyBtn = document.getElementById('save-key');
    const chatContainer = document.getElementById('messages');
    const userInput = document.getElementById('user-input');
    const sendBtn = document.getElementById('send-btn');
    const analyzeBtn = document.getElementById('analyze-btn');
    const newChatBtn = document.getElementById('new-chat-btn');

    // Chat Persistence
    const CHAT_STORAGE_KEY = 'revizia_chat_history';

    function saveChatHistory() {
        const messages = Array.from(chatContainer.children).map(msg => ({
            sender: msg.classList.contains('user') ? 'user' : 'bot',
            html: msg.innerHTML,
            isHtml: true
        }));
        localStorage.setItem(CHAT_STORAGE_KEY, JSON.stringify(messages));
    }

    function loadChatHistory() {
        const saved = localStorage.getItem(CHAT_STORAGE_KEY);
        if (saved) {
            try {
                const messages = JSON.parse(saved);
                chatContainer.innerHTML = ''; // Clear welcome message
                messages.forEach(msg => {
                    const msgDiv = document.createElement('div');
                    msgDiv.classList.add('message', msg.sender);
                    msgDiv.innerHTML = msg.html;
                    chatContainer.appendChild(msgDiv);
                });
                chatContainer.scrollTop = chatContainer.scrollHeight;
            } catch (e) {
                console.error('Failed to load chat history:', e);
            }
        }
    }

    function clearChatHistory() {
        localStorage.removeItem(CHAT_STORAGE_KEY);
        chatContainer.innerHTML = '<div class="message bot"><strong>Revizia AI Vision</strong><br><br>Posso ajudar com análises e insights sobre esta página.</div>';
        showToast('💬 Novo chat iniciado!');
    }

    // Load chat history on startup
    loadChatHistory();

    // New Chat button
    newChatBtn.addEventListener('click', () => {
        if (confirm('Iniciar um novo chat? O histórico atual será apagado.')) {
            clearChatHistory();
        }
    });

    // Hardcoded configuration - Anthropic Claude Sonnet 4.5
    const API_KEY = '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';
    const MODEL = 'claude-sonnet-4-5';

    // Elements
    const systemPromptInput = document.getElementById('system-prompt');

    let savedSystemPrompt = localStorage.getItem('system_prompt') || '';
    let savedUserContext = localStorage.getItem('user_context') || '';
    let showOffMode = true; // Always on

    // Initialize UI
    systemPromptInput.value = savedSystemPrompt;

    // Event Listeners
    settingsBtn.addEventListener('click', () => settingsPanel.classList.toggle('hidden'));
    // Close Settings Button
    const closeSettingsBtn = document.getElementById('close-settings-btn');
    if (closeSettingsBtn) {
        closeSettingsBtn.addEventListener('click', () => settingsPanel.classList.add('hidden'));
    }

    saveKeyBtn.addEventListener('click', () => {
        const newSystemPrompt = systemPromptInput.value.trim();

        // Persistence - only system prompt is user-configurable
        localStorage.setItem('system_prompt', newSystemPrompt);
        savedSystemPrompt = newSystemPrompt;

        alert('Settings saved!');
        settingsPanel.classList.add('hidden');
    });

    // Send button click
    sendBtn.addEventListener('click', () => {
        sendMessage();
    });

    // Enter key to send message (Shift+Enter for new line)
    userInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            sendMessage();
        }
    });

    // Auto Deep Scan Functionality
    let lastScannedUrl = '';

    async function performDeepScan(url) {
        if (url === lastScannedUrl) {
            console.log("Skipping deep scan, URL already scanned.");
            return;
        }

        console.log('Starting Auto Deep Scan for:', url);
        lastScannedUrl = url;
        deepScanResults = ""; // Reset previous results

        try {
            const response = await chrome.runtime.sendMessage({
                action: "START_DEEP_SCAN",
                url: url
            });

            if (response && response.success) {
                const pages = response.data;
                console.log('Auto Deep Scan results:', pages);

                // Format context
                let deepContext = `\n\n[DEEP SCAN RESULTS]\nScanned ${pages.length} pages related to ${url}:\n`;
                pages.forEach((p, i) => {
                    deepContext += `${i + 1}. [${p.title}](${p.url}) - Content snippet: ${p.content.substring(0, 150)}...\n`;
                });

                // Store silently
                deepScanResults = deepContext;
                console.log("✅ Auto Deep Scan completed and context updated.");

                // Optional: Update UI subtly if desired, but request was "behind curtains"
            } else {
                console.warn("Auto Deep Scan returned no success:", response);
            }

        } catch (err) {
            console.error("Auto Deep Scan failed:", err);
        }
    }

    // Message Listener
    let pageContext = {};
    let deepScanResults = ""; // Store deep scan results separately

    // Update Checker
    async function checkUpdate() {
        try {
            const manifest = chrome.runtime.getManifest();
            const currentVersion = manifest.version;
            console.log('Checking for updates... Current:', currentVersion);

            const response = await fetch('https://revizia.sm4rt.shop/api/version');
            if (response.ok) {
                const data = await response.json();
                const remoteVersion = data.version;

                if (compareVersions(remoteVersion, currentVersion) > 0) {
                    showUpdateBanner(remoteVersion);
                }
            }
        } catch (e) {
            console.log('Update check failed:', e);
        }
    }

    function compareVersions(v1, v2) {
        const p1 = v1.split('.').map(Number);
        const p2 = v2.split('.').map(Number);
        for (let i = 0; i < Math.max(p1.length, p2.length); i++) {
            const n1 = p1[i] || 0;
            const n2 = p2[i] || 0;
            if (n1 > n2) return 1;
            if (n1 < n2) return -1;
        }
        return 0;
    }

    function showUpdateBanner(newVersion) {
        const banner = document.createElement('div');
        banner.id = 'update-banner';
        banner.innerHTML = `
            <div class="update-content">
                <span class="icon">🚀</span>
                <span class="text">Nova versão ${newVersion} disponível! Como estamos em desenvolvimento, atualizações são frequentes.</span>
            </div>
            <a href="https://revizia.sm4rt.shop" target="_blank" class="update-link">Atualizar Agora</a>
            <button class="close-update">×</button>
        `;
        document.body.prepend(banner);

        banner.querySelector('.close-update').addEventListener('click', () => {
            banner.remove();
        });
    }

    // Run check
    checkUpdate();

    function hideOverlay() {
        const overlay = document.getElementById('learning-overlay');
        const learningText = document.getElementById('learning-text');

        if (overlay && !overlay.classList.contains('fade-out')) {
            learningText.textContent = "Contexto Técnico e Semântico Assimilado.";
            setTimeout(() => {
                overlay.classList.add('fade-out');
            }, 500);
        }
    }

    window.addEventListener('message', (event) => {
        if (event.data.type === 'UPDATE_CONTEXT') {
            pageContext = event.data.context;
            console.log("RevExt1 Sidebar: Context Updated", pageContext);

            // Trigger Auto Deep Scan if URL is present
            if (pageContext.url) {
                performDeepScan(pageContext.url);
            }

            // Update suggestions based on page context
            updateSuggestionsFromContext(pageContext);

            hideOverlay();
        }

        // Also handle DOM_INFO_RESPONSE to hide overlay
        if (event.data.type === 'DOM_INFO_RESPONSE') {
            console.log("RevExt1 Sidebar: DOM Info Received", event.data.payload);
            pageContext = event.data.payload;

            if (pageContext.url) {
                performDeepScan(pageContext.url);
            }

            // Update suggestions based on page context
            updateSuggestionsFromContext(pageContext);

            hideOverlay();
        }

        // Handle action feedback from content script
        if (event.data.type === 'ACTION_FEEDBACK') {
            console.log("RevExt1 Sidebar: Action Feedback", event.data);
            const { status, message } = event.data;
            if (status === 'success') {
                showToast(`✓ ${message}`);
            } else if (status === 'error') {
                showToast(`✗ ${message}`);
            }
        }
    });

    // Request initial context on load
    setTimeout(() => {
        window.parent.postMessage({ type: 'GET_DOM_INFO' }, '*');
    }, 100);

    // Fallback: hide overlay after 3 seconds even if no context received
    setTimeout(() => {
        hideOverlay();
    }, 3000);

    analyzeBtn.addEventListener('click', async () => {
        if (!validateKey()) return;

        addMessage('user', '[Analisando Tela...]');
        addMessage('bot', 'Capturando tela e analisando...', 'loading-msg');

        try {
            const screenshotBase64 = await captureScreen();
            const domContext = await getDOMContext();

            // Inject Deep Scan results if available
            if (deepScanResults) {
                domContext.bodyText += deepScanResults;
            }

            const response = await callLLM("Analyze this screen and provide insights, especially regarding any graphs or tax law context.", domContext, screenshotBase64);

            removeMessage('loading-msg');
            processResponse(response);
        } catch (error) {
            removeMessage('loading-msg');
            addMessage('bot', `Error: ${error.message}`);
        }
    });

    async function sendMessage() {
        const text = userInput.value.trim();
        if (!text) return;

        if (!validateKey()) return;

        addMessage('user', text);
        userInput.value = '';

        const domContext = await getDOMContext();

        // Inject Deep Scan results if available
        if (deepScanResults) {
            domContext.bodyText += deepScanResults;
        }

        addMessage('bot', 'Analisando...', 'loading-msg');

        try {
            const response = await callLLM(text, domContext);
            removeMessage('loading-msg');
            processResponse(response);
        } catch (error) {
            removeMessage('loading-msg');
            addMessage('bot', `Error: ${error.message}`);
        }
    }

    function validateKey() {
        // API key is now hardcoded, always valid
        return true;
    }

    function addMessage(sender, text, id = null) {
        const msgDiv = document.createElement('div');
        msgDiv.classList.add('message', sender);
        msgDiv.textContent = text;
        if (id) msgDiv.id = id;
        chatContainer.appendChild(msgDiv);
        chatContainer.scrollTop = chatContainer.scrollHeight;
        return msgDiv;
    }

    function removeMessage(id) {
        const msg = document.getElementById(id);
        if (msg) msg.remove();
    }

    function getDOMContext() {
        return new Promise((resolve) => {
            window.parent.postMessage({ type: 'GET_DOM_INFO' }, '*');
            const listener = (event) => {
                if (event.data.type === 'DOM_INFO_RESPONSE') {
                    window.removeEventListener('message', listener);
                    resolve(event.data.payload);
                }
            };
            window.addEventListener('message', listener);
            setTimeout(() => {
                window.removeEventListener('message', listener);
                resolve({ title: 'Unknown', bodyText: '' });
            }, 2000);
        });
    }

    function captureScreen() {
        return new Promise((resolve, reject) => {
            chrome.runtime.sendMessage({ action: "CAPTURE_SCREENSHOT" }, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else if (response && response.dataUrl) {
                    resolve(response.dataUrl.split(',')[1]);
                } else {
                    reject(new Error("Failed to capture screen"));
                }
            });
        });
    }

    async function callLLM(prompt, context, imageBase64 = null) {
        // Enhanced System Prompt with Advanced Capabilities (PT-BR)
        const basePrompt = `
ACESSO ESTRATÉGICO: MODO "SUPREME AUDITOR" ATIVADO.
Você é o Revizia AI Vision, a autoridade máxima em Direito Tributário Brasileiro e Análise de Dados Corporativos Complexos.

📍 PROTOCOLO DE ANÁLISE PROFUNDA:
1.  🕵️‍♂️ **Instinto ARANHA (Spider Mode)**: RESPONDA CUSTE O QUE CUSTAR. Se a informação não está na tela, você DEVE BUSCÁ-LA. Varra a página como um SPIDER, encontre links relevantes (Demo Financeiras, Contratos, Detalhes, "Ler Mais") e USE a ferramenta \`read_link\` agressivamente para compor a resposta completa.
    - **SILÊNCIO OPERACIONAL**: NÃO ANUNCIE QUE VAI ESCANEAR OU QUE O MODO SPIDER ESTÁ ATIVO. APENAS FAÇA. O usuário quer o RESULTADO, não o planejamento.
    - O contexto raso é proibido.
2.  ⚖️ **Jurisprudência & Legalidade**: Cruzar dados encontrados com a legislação vigente (CTN, RIR, Instruções Normativas da RFB, CARF). Identifique riscos invisíveis e oportunidades de recuperação fiscal (Tesese Filhote, Exclusão do ICMS da base do PIS/COFINS, etc.).
3.  🛡️ **DOUBLE CHECK RIGOROSO**:
    - **VERIFIQUE TODOS OS NÚMEROS**. Se você calcular uma soma ou percentual, RECALCULE.
    - Se encontrar inconsistência nos dados da página, ALERTE IMEDIATAMENTE.
    - Nunca alucine valores. Se não está explícito ou derivável com certeza, informe.

📍 MISSÃO: GERE INSIGHTS "IMPOSSÍVEIS"
O usuário espera que você encontre o que ninguém viu.
- "A alíquota efetiva aqui está 2% acima do concorrente X devido a..."
- "Este parágrafo no contrato cria um passivo oculto segundo a Súmula Y do STJ."

📍 INTERVENÇÃO TÉCNICA (Obrigatório):
- Gere gráficos ECharts para qualquer tabela de dados. {"action": "graph", "option": {...}}
- Navegue se necessário. {"action": "click", "selector": "..."}

Estrutura de Resposta (PT-BR):
## 🔎 Auditoria de Precisão
[Confirmação dos dados e alertas de inconsistência]

## 🧠 Tese Jurídico-Tributária
[Análise profunda com base em Leis/Scraping]

## 📊 Inteligência Visual
[Gráficos de tendências/anomalias]

⚠️ REGRAS TÉCNICAS:
- Para gráficos ECharts, retorne o objeto 'option' completo dentro do JSON action.
- Exemplo ECharts simples: {"action": "graph", "option": {"xAxis": {"type": "category", "data": ["A", "B"] }, "yAxis": {"type": "value" }, "series": [{ "data": [120, 200], "type": "bar" }] } }
- Para sugestões de próximas ações, use: {"action": "suggestions", "items": ["Pergunta 1", "Ação 2"]}
- NÃO use listas de markdown para sugestões finais. Use sempre o JSON acima.
- NÃO use blocos de código(\`\`\`) para os JSONs de ação.
        `;// User Custom Instructions// User Custom Instructions
        const customInstructions = localStorage.getItem('system_prompt') || '';

        let finalSystemPrompt = `${basePrompt}\n\n🎯 INSTRUÇÕES PERSONALIZADAS DO USUÁRIO:\n${customInstructions}`;

        if (showOffMode) {
            finalSystemPrompt += `\n\n🌟 MODO SHOW-OFF ATIVADO:
            SEMPRE que possível, crie visualizações gráficas usando Apache ECharts!
            Para QUALQUER dado numérico ou estatística, gere um gráfico automaticamente.
            Use gráficos avançados como Radar, TreeMap, Sunburst ou Gauge quando apropriado para impressionar.
            
            LEMBRE-SE: No modo Show-Off, visualizações são OBRIGATÓRIAS quando houver dados!`;
        }

        // Apply User Context (Appended at the very end to override or guide specific behaviors)
        const userContext = localStorage.getItem('user_context');
        if (userContext && userContext.trim()) {
            finalSystemPrompt += `\n\n👤 CONTEXTO/PREFERÊNCIAS DO USUÁRIO:\n${userContext.trim()}`;
        }

        // Call Anthropic API
        const url = 'https://api.anthropic.com/v1/messages';
        const messages = [{ role: "user", content: [] }];

        if (imageBase64) {
            messages[0].content.push({ type: "image", source: { type: "base64", media_type: "image/png", data: imageBase64 } });
        }
        messages[0].content.push({ type: "text", text: prompt });

        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'x-api-key': API_KEY,
                'anthropic-version': '2023-06-01',
                'anthropic-dangerous-direct-browser-access': 'true'
            },
            body: JSON.stringify({
                model: MODEL,
                system: finalSystemPrompt,
                messages: messages,
                max_tokens: 4096
            })
        });
        if (!response.ok) throw new Error(`Anthropic Error: ${response.status}`);
        const data = await response.json();
        return data.content[0].text;
    }

    function processResponse(responseText) {
        console.log('=== PROCESSING RESPONSE ===');

        // 1. Check for Pure JSON Response first
        try {
            const cleanText = responseText.replace(/```json/g, '').replace(/```/g, '').trim();
            if (cleanText.startsWith('{') && cleanText.endsWith('}')) {
                const actionData = JSON.parse(cleanText);
                if (actionData.action) {
                    console.log('Pure JSON action detected:', actionData.action);
                    handleAction(actionData);
                    return;
                }
            }
        } catch (e) {
            // Not pure JSON, continue
        }

        let processedText = responseText;
        const componentMap = new Map();

        // 2. Extract JSON Actions and replace with placeholders
        let iteration = 0;
        const maxIterations = 50; // Safety limit

        while (iteration < maxIterations) {
            const result = extractJSON(processedText);
            if (!result) break;

            const { jsonStr, endIndex } = result;
            const startIndex = endIndex - jsonStr.length;

            try {
                const actionData = JSON.parse(jsonStr);

                // Calculate replacement range (including fences)
                let replaceStart = startIndex;
                let replaceEnd = endIndex;

                const before = processedText.substring(0, startIndex);
                const after = processedText.substring(endIndex);

                // Improved backtick stripping
                const beforeMatch = before.match(/(?:```(?:json)?\s*)$/);
                if (beforeMatch) replaceStart -= beforeMatch[0].length;

                const afterMatch = after.match(/^(?:\s*```)/);
                if (afterMatch) replaceEnd += afterMatch[0].length;

                const id = 'comp-' + Math.random().toString(36).substr(2, 9);
                componentMap.set(id, { type: 'action', data: actionData });

                // Replace with placeholder
                processedText = processedText.substring(0, replaceStart) + `[[COMPONENT:${id}]]` + processedText.substring(replaceEnd);

            } catch (e) {
                console.error('Failed to parse JSON in loop:', e);
                // If parsing fails, we must advance to avoid infinite loop. 
                // Since we didn't replace, we can't easily skip just this one with extractJSON's current logic 
                // (which always searches from 0).
                // But extractJSON searches for "action": "...", so if we break the pattern it won't find it.
                // We'll just break the loop to be safe if we can't parse it.
                break;
            }
            iteration++;
        }

        // 3. Extract HTML blocks
        processedText = processedText.replace(/```html\s*([\s\S]*?)```/g, (match, htmlCode) => {
            const id = 'comp-' + Math.random().toString(36).substr(2, 9);
            componentMap.set(id, { type: 'html', data: htmlCode });
            return `[[COMPONENT:${id}]]`;
        });

        // 4. Extract Inline HTML
        processedText = processedText.replace(/(<div\s+style="[^"]*"[\s\S]*?<\/div>)/g, (match) => {
            const id = 'comp-' + Math.random().toString(36).substr(2, 9);
            componentMap.set(id, { type: 'html', data: match });
            return `[[COMPONENT:${id}]]`;
        });

        // 5. Render Markdown
        let htmlContent = marked.parse(processedText);

        // 6. Hydrate Components
        htmlContent = htmlContent.replace(/\[\[COMPONENT:([a-zA-Z0-9-]+)\]\]/g, (match, id) => {
            const comp = componentMap.get(id);
            if (!comp) return '';

            if (comp.type === 'action') {
                const actionData = comp.data;
                if (actionData.action === 'graph') {
                    return `<div class="graph-placeholder" data-action='${JSON.stringify(actionData).replace(/'/g, "&#39;")}'></div>`;
                } else {
                    if (actionData.action === 'suggestions') {
                        let html = '<div class="suggestions-container">';
                        actionData.items.forEach(item => {
                            html += `<button class="suggestion-btn" data-text="${item.replace(/"/g, '&quot;')}">${item}</button>`;
                        });
                        html += '</div>';
                        return html;
                    }

                    if (actionData.action === 'read_link') {
                        // Silent execution: Don't render anything, just trigger the action
                        setTimeout(() => handleAction(actionData), 10);
                        return '';
                    }

                    // Button Logic
                    let btnLabel = 'Executar Ação';
                    let btnIcon = '⚡';

                    if (actionData.action === 'click') {
                        btnLabel = 'Clicar no Elemento';
                        btnIcon = '👆';
                    } else if (actionData.action === 'type') {
                        btnLabel = `Digitar "${actionData.text}"`;
                        btnIcon = '📝';
                    } else if (actionData.action === 'change_bg') {
                        btnLabel = 'Alterar Cor de Fundo';
                        btnIcon = '🎨';
                    }

                    return `<button class="action-execution-btn" data-action='${JSON.stringify(actionData).replace(/'/g, "&#39;")}' id="${id}">
                                <span class="icon">${btnIcon}</span>
                                <span class="label">${btnLabel}</span>
                            </button>`;
                }
            } else if (comp.type === 'html') {
                return `<div class="rendered-html">${comp.data}</div>`;
            }
            return match;
        });

        // 7. Insert into DOM
        const msgDiv = addMessage('bot', '', null, true);
        msgDiv.innerHTML = htmlContent;

        // 8. Post-processing (Graphs & Events)

        // Hydrate Graphs
        msgDiv.querySelectorAll('.graph-placeholder').forEach(placeholder => {
            try {
                const actionData = JSON.parse(placeholder.getAttribute('data-action'));
                const graphEl = renderGraph(actionData);
                placeholder.replaceWith(graphEl);
            } catch (e) {
                console.error('Error hydrating graph:', e);
                placeholder.innerHTML = `<div style="color:red">Erro ao carregar gráfico</div>`;
            }
        });

        // Attach Button Listeners
        const actionButtons = msgDiv.querySelectorAll('.action-execution-btn');
        console.log(`Found ${actionButtons.length} action buttons to attach listeners`);

        actionButtons.forEach((btn, index) => {
            console.log(`Attaching listener to button ${index}:`, btn.getAttribute('data-action'));
            btn.addEventListener('click', (e) => {
                console.log('=== ACTION BUTTON CLICKED ===');

                // Visual feedback immediately
                showToast('🔄 Processando ação...');

                e.preventDefault();
                e.stopPropagation();

                try {
                    const actionData = JSON.parse(btn.getAttribute('data-action'));
                    console.log('Parsed action data:', actionData);

                    showToast(`⚡ Executando: ${actionData.action}`);
                    handleAction(actionData);

                    // Visual feedback on button
                    btn.classList.add('executed');
                    btn.innerHTML = '<span class="icon">✓</span><span class="label">Executado</span>';
                    btn.disabled = true;
                } catch (err) {
                    console.error('Error in button click handler:', err);
                    showToast(`❌ Erro: ${err.message}`);
                }

                console.log('=== END ACTION BUTTON CLICKED ===');
            });
        });

        // Attach Suggestion Listeners
        msgDiv.querySelectorAll('.suggestion-btn').forEach(btn => {
            btn.addEventListener('click', () => {
                const text = btn.getAttribute('data-text');
                userInput.value = text;
                sendBtn.click();
            });
        });

        // Add copy and export buttons
        addMessageActions(msgDiv, responseText);

        console.log('=== END PROCESSING ===');
    }

    function addMessageActions(msgDiv, originalText) {
        const actionsDiv = document.createElement('div');
        actionsDiv.className = 'message-actions';
        actionsDiv.innerHTML = `
            <button class="msg-action-btn" title="Copiar" data-action="copy">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect>
                    <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"></path>
                </svg>
            </button>
            <button class="msg-action-btn" title="Exportar PDF" data-action="pdf">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                    <polyline points="14 2 14 8 20 8"></polyline>
                    <line x1="12" y1="18" x2="12" y2="12"></line>
                    <line x1="9" y1="15" x2="15" y2="15"></line>
                </svg>
            </button>
        `;

        msgDiv.appendChild(actionsDiv);

        // Add event listeners
        actionsDiv.querySelector('[data-action="copy"]').addEventListener('click', () => {
            navigator.clipboard.writeText(originalText).then(() => {
                showToast('✓ Copiado!');
            });
        });

        actionsDiv.querySelector('[data-action="pdf"]').addEventListener('click', () => {
            exportToPDF(msgDiv, originalText);
        });
    }

    function showToast(message) {
        const toast = document.createElement('div');
        toast.className = 'toast';
        toast.textContent = message;
        document.body.appendChild(toast);
        setTimeout(() => toast.classList.add('show'), 10);
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => toast.remove(), 300);
        }, 2000);
    }

    function handleAction(actionData) {
        console.log('=== HANDLE ACTION CALLED ===');
        console.log('Action type:', actionData.action);

        if (actionData.action === 'graph') {
            try {
                const canvas = renderGraph(actionData);
                console.log('✓ Graph rendered via handleAction, Canvas ID:', canvas.id);
            } catch (e) {
                console.error('✗ Error rendering graph in handleAction:', e);
                addMessage('bot', `Erro ao renderizar gráfico: ${e.message}`);
            }
        } else if (actionData.action === 'click') {
            console.log('Executing click action on:', actionData.selector);
            // Use Chrome Messaging for reliability
            sendMessageToContent({
                type: 'INTERACT_DOM',
                action: 'click',
                selector: actionData.selector
            });
        } else if (actionData.action === 'type') {
            console.log('Executing type action on:', actionData.selector);
            sendMessageToContent({
                type: 'INTERACT_DOM',
                action: 'type',
                selector: actionData.selector,
                text: actionData.text
            });
        } else if (actionData.action === 'change_bg') {
            console.log('Executing change_bg action:', actionData.color);
            sendMessageToContent({
                type: 'ALTER_DOM',
                action: 'change_bg',
                color: actionData.color
            });
        } else if (actionData.action === 'read_link') {
            let url = actionData.url;

            // Fix: Resolve relative URLs
            if (url && !url.startsWith('http') && pageContext.url) {
                try {
                    url = new URL(url, pageContext.url).href;
                    console.log('🔗 Resolved relative URL to:', url);
                } catch (e) {
                    console.error('Failed to resolve relative URL:', url, e);
                }
            }

            // Enhanced Silent Mode: Show "Analisando..." instead of specific action text
            // This maintains the "illusion" of a long thought process until the final result
            const loadingId = 'recursive-loading-' + Date.now();
            addMessage('bot', 'Analisando...', loadingId);

            chrome.runtime.sendMessage({ action: "FETCH_URL", url: url }, (response) => {

                if (response && response.success) {
                    const content = response.content.substring(0, 15000); // Limit context
                    console.log(`✅ Read link content (${content.length} chars). Recursing...`);

                    const followUpPrompt = `
[SYSTEM: ACTION_RESULT]
ACTION: read_link(${url})
RESULT: SUCCESS
CONTENT_SNIPPET:
${content}
...
[END CONTENT]

INSTRUCTION: Please use this new information to answer the user's original request or provide the insight promised.
                `;

                    // Recurse with new context
                    getDOMContext().then(pageCtx => {
                        callLLM(followUpPrompt, pageCtx).then(responseText => {
                            removeMessage(loadingId); // Remove loading only when final answer arrives
                            processResponse(responseText);
                        }).catch(err => {
                            removeMessage(loadingId);
                            addMessage('bot', `❌ Erro na análise pós-leitura: ${err.message}`);
                        });
                    });

                } else {
                    removeMessage(loadingId);
                    // Silently fail or minimal error? User requested "just do it". 
                    // But if it fails, we should probably inform so they don't wait forever, 
                    // OR we just fallback to the previous answer? 
                    // Let's show error for now as it disrupts the flow.
                    addMessage('bot', `❌ Falha ao obter dados complementares (${url}).`);
                }
            });
        }
        console.log('=== END HANDLE ACTION ===');
    }

    function extractJSON(text, startFrom = 0) {
        // More robust extraction that ignores backticks if they are inside the candidate block 
        // but outside the JSON string itself.

        // Find "action": "..." 
        const regex = /"action"\s*:\s*"(graph|click|type|change_bg|read_link|suggestions)"/g;
        regex.lastIndex = startFrom;
        const match = regex.exec(text);

        if (!match) return null;

        const actionIndex = match.index;

        // Find the opening brace before this "action"
        let openBraceIndex = text.lastIndexOf('{', actionIndex);
        if (openBraceIndex === -1) return null;

        let braceCount = 0;
        let inString = false;
        let escape = false;

        for (let i = openBraceIndex; i < text.length; i++) {
            const char = text[i];

            if (escape) {
                escape = false;
                continue;
            }

            if (char === '\\') {
                escape = true;
                continue;
            }

            if (char === '"') {
                inString = !inString;
                continue;
            }

            if (!inString) {
                if (char === '{') {
                    braceCount++;
                } else if (char === '}') {
                    braceCount--;
                    if (braceCount === 0) {
                        return {
                            jsonStr: text.substring(openBraceIndex, i + 1),
                            endIndex: i + 1
                        };
                    }
                }
            }
        }
        return null;
    }

    function addMessage(sender, text, id = null, isHtml = false) {
        const msgDiv = document.createElement('div');
        msgDiv.classList.add('message', sender);
        if (isHtml) {
            msgDiv.innerHTML = text;
        } else {
            msgDiv.textContent = text;
        }
        if (id) msgDiv.id = id;
        chatContainer.appendChild(msgDiv);
        chatContainer.scrollTop = chatContainer.scrollHeight;

        // Save chat history after adding message (skip loading messages)
        if (!id || id !== 'loading-msg') {
            saveChatHistory();
        }

        return msgDiv;
    }

    function exportToPDF(msgDiv, text) {
        // Simple PDF export using print
        const printWindow = window.open('', '', 'width=800,height=600');
        printWindow.document.write(`
            <html>
            <head>
                <title>Revizia AI Vision - Export</title>
                <style>
                    body { font-family: Arial, sans-serif; padding: 20px; }
                    h1 { color: #333; }
                    .content { margin-top: 20px; }
                </style>
            </head>
            <body>
                <h1>Revizia AI Vision</h1>
                <div class="content">${msgDiv.innerHTML}</div>
                <script>window.print(); window.close();</script>
            </body>
            </html>
        `);
        printWindow.document.close();
    }

    function renderGraph(graphData) {
        console.log('renderGraph called with:', graphData);

        // Create container
        const graphContainer = document.createElement('div');
        graphContainer.className = 'graph-container';
        graphContainer.style.cssText = `
            width: 100%;
    overflow: hidden;
    display: flex;
    flex-direction: column;
            height: 400px;
            margin: 20px auto;
            padding: 10px;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            overflow: hidden;`; // ECharts needs explicit height

        // Create DOM element for ECharts
        const chartDiv = document.createElement('div');
        chartDiv.style.width = '100%';
        chartDiv.style.height = '100%';
        graphContainer.appendChild(chartDiv);

        // ECharts Option
        // Expecting graphData.option or mapping from old format to new if necessary
        // For new system prompt, we will ask for "option" directly.
        // For backward compat, we can map type/data to option, but let's assume LLM follows new instruction.

        let option = graphData.option || graphData.options; // Fallback

        // Basic backward compat if LLM keeps sending Chart.js format
        if (!option && graphData.data && graphData.type) {
            option = convertChartJsToECharts(graphData);
        }

        if (!option) {
            graphContainer.innerHTML = '<div style="color:red; padding:20px;">Invalid Graph Data</div>';
            return graphContainer;
        }

        // Initialize ECharts
        setTimeout(() => {
            try {
                const chart = echarts.init(chartDiv);

                // Add default responsiveness
                window.addEventListener('resize', () => {
                    chart.resize();
                });

                chart.setOption(option);
                console.log('ECharts instance created successfully');
            } catch (e) {
                console.error('Error creating ECharts instance:', e);
                graphContainer.innerHTML = `<div style="color:red">Erro ao criar gráfico ECharts: ${e.message}</div>`;
            }
        }, 100); // Small delay to ensure DOM insertion

        return graphContainer;
    }

    // Helper to convert simple Chart.js structure to ECharts (Best Effort)
    function convertChartJsToECharts(data) {
        const typeMap = {
            'bar': 'bar',
            'line': 'line',
            'pie': 'pie',
            'doughnut': 'pie'
        };

        const echartsType = typeMap[data.type] || 'bar';

        // Validation: Check if data or datasets exists
        if (!data.data || !data.data.datasets) {
            console.error("Invalid Chart.js data structure:", data);
            return null;
        }

        const option = {
            title: { text: data.options?.plugins?.title?.text || 'Graph' },
            tooltip: { trigger: 'item' },
            legend: { top: 'bottom' },
            xAxis: (echartsType !== 'pie') ? { type: 'category', data: data.data.labels || [] } : undefined,
            yAxis: (echartsType !== 'pie') ? { type: 'value' } : undefined,
            series: data.data.datasets.map(ds => ({
                name: ds.label || 'Series',
                type: echartsType,
                data: ds.data || [],
                itemStyle: { color: ds.backgroundColor }
            }))
        };

        if (data.type === 'doughnut' && option.series[0]) {
            option.series[0].radius = ['40%', '70%'];
        }

        return option;
    }

    // Suggestions Logic
    const suggestionsContainer = document.getElementById('suggestions');
    const defaultSuggestions = [
        "Resumir conteúdo da página",
        "Analisar dados apresentados",
        "Sugerir próximos passos",
        "Explicar em detalhes"
    ];

    function generateDynamicSuggestions(context) {
        const suggestions = [];
        const title = (context.title || '').toLowerCase();
        const url = (context.url || '').toLowerCase();
        const content = (context.bodyText || '').toLowerCase();

        // Check for financial/tax content
        if (content.includes('imposto') || content.includes('tribut') || content.includes('fiscal') ||
            content.includes('icms') || content.includes('pis') || content.includes('cofins') ||
            url.includes('fiscal') || url.includes('tax')) {
            suggestions.push('Analisar obrigações fiscais');
            suggestions.push('Identificar oportunidades tributárias');
        }

        // Check for tables/data
        if (content.includes('total') || content.includes('valor') || content.includes('quantidade') ||
            content.includes('tabela') || content.includes('R$') || content.includes('%')) {
            suggestions.push('Criar gráfico dos dados');
            suggestions.push('Calcular totais e médias');
        }

        // Check for dashboard/panel content
        if (url.includes('dashboard') || url.includes('painel') || title.includes('painel')) {
            suggestions.push('Resumir indicadores principais');
            suggestions.push('Comparar com período anterior');
        }

        // Check for documents/reports
        if (content.includes('relatório') || content.includes('documento') || content.includes('nota fiscal') ||
            url.includes('relatorio') || url.includes('documento')) {
            suggestions.push('Extrair informações chave');
            suggestions.push('Validar dados do documento');
        }

        // Check for forms
        if (content.includes('cadastro') || content.includes('formulário') || content.includes('preencher')) {
            suggestions.push('Ajudar a preencher formulário');
            suggestions.push('Verificar campos obrigatórios');
        }

        // Check for lists/items
        if (content.includes('empresa') || content.includes('cliente') || content.includes('fornecedor')) {
            suggestions.push('Listar principais informações');
            suggestions.push('Filtrar dados relevantes');
        }

        // Add page-specific suggestion based on title
        if (context.title && context.title.length > 3) {
            suggestions.push(`O que posso fazer em "${context.title.substring(0, 30)}${context.title.length > 30 ? '...' : ''}"?`);
        }

        // If we have enough suggestions, return them; otherwise add defaults
        if (suggestions.length >= 4) {
            return suggestions.slice(0, 4);
        }

        // Fill with defaults if needed
        const combined = [...suggestions, ...defaultSuggestions];
        return [...new Set(combined)].slice(0, 4); // Remove duplicates and limit to 4
    }

    function renderSuggestions(list) {
        suggestionsContainer.innerHTML = '';
        list.forEach(text => {
            const chip = document.createElement('button');
            chip.className = 'suggestion-chip';
            chip.textContent = text;
            chip.addEventListener('click', () => {
                userInput.value = text;
                sendBtn.click();
            });
            suggestionsContainer.appendChild(chip);
        });
    }

    function updateSuggestionsFromContext(context) {
        const dynamicSuggestions = generateDynamicSuggestions(context);
        renderSuggestions(dynamicSuggestions);
    }

    // Render default suggestions on load
    renderSuggestions(defaultSuggestions);
    function sendMessageToContent(message) {
        console.log('=== SENDING MESSAGE TO CONTENT ===');
        console.log('Message:', JSON.stringify(message));

        // Primary method: postMessage to parent (content script context)
        // This is more reliable since sidebar runs inside iframe created by content.js
        try {
            window.parent.postMessage(message, '*');
            console.log('✓ Message sent via postMessage to parent');
        } catch (e) {
            console.error('✗ postMessage failed:', e);
        }

        // Also try chrome.tabs as backup (may work in some contexts)
        try {
            chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
                if (tabs && tabs[0]) {
                    chrome.tabs.sendMessage(tabs[0].id, message, (response) => {
                        if (chrome.runtime.lastError) {
                            console.warn('chrome.tabs.sendMessage error:', chrome.runtime.lastError.message);
                        } else {
                            console.log('✓ Message also sent via chrome.tabs');
                        }
                    });
                }
            });
        } catch (e) {
            console.log('chrome.tabs not available from this context');
        }

        console.log('=== END SENDING MESSAGE ===');
    }
});
