// Revizia High-Performance Offscreen Core

// Listen for messages from the background service worker or sidebar
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.target !== "OFFSCREEN") return;

    console.log("[Offscreen] Received message:", message);

    switch (message.type) {
        case "PARSE_HTML":
            try {
                const result = parseHTML(message.data.html, message.data.url, message.data.depth);
                sendResponse({ success: true, data: result });
            } catch (err) {
                console.error("[Offscreen] Parse error:", err);
                sendResponse({ success: false, error: err.toString() });
            }
            break;

        case "PING":
            sendResponse("PONG");
            break;

        default:
            console.warn("[Offscreen] Unknown message type:", message.type);
    }
});

// Parse HTML content (DOM operations only, no network requests)
function parseHTML(html, url, depth) {
    console.log(`[Offscreen] Parsing HTML for ${url} (Depth: ${depth})`);

    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');

    // Extract useful content (title + main text)
    const title = doc.title || url;

    // Simple content extraction: remove scripts/styles, get text
    const scripts = doc.querySelectorAll('script, style, noscript, svg, img, video');
    scripts.forEach(s => s.remove());
    const textContent = doc.body ? doc.body.innerText.replace(/\s+/g, ' ').trim().substring(0, 2000) : '';

    // Extract links for background to process
    const links = [];
    const linkElements = doc.querySelectorAll('a[href]');
    for (const link of linkElements) {
        const href = link.getAttribute('href');
        if (href) {
            links.push(href);
        }
    }

    return {
        url: url,
        title: title,
        content: textContent,
        depth: depth,
        links: links.slice(0, 50) // Limit to 50 links per page
    };
}
