if (typeof sidebarOpen === 'undefined') {
    var sidebarOpen = false;
    var sidebarIframe = null;
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === "toggleSidebar") {
        toggleSidebar();
    } else if (request.type === "INTERACT_DOM" || request.type === "ALTER_DOM") {
        handleDOMInteraction(request);
    }
});

function toggleSidebar() {
    if (sidebarOpen) {
        closeSidebar();
    } else {
        openSidebar();
    }
}

function openSidebar() {
    if (!sidebarIframe) {
        sidebarIframe = document.createElement('iframe');
        sidebarIframe.style.position = 'fixed';
        sidebarIframe.id = 'revizia-sidebar';
        sidebarIframe.src = chrome.runtime.getURL('sidebar.html');
        sidebarIframe.style.cssText = `
            position: fixed;
            top: 0;
            right: 0;
            width: 400px;
            height: 100%;
            border: none;
            z-index: 2147483647;
            box-shadow: -2px 0 10px rgba(0,0,0,0.3);
        `;

        // Create resize handle
        const resizeHandle = document.createElement('div');
        resizeHandle.id = 'revizia-resize-handle';
        resizeHandle.style.cssText = `
            position: fixed;
            top: 0;
            right: 400px;
            width: 6px;
            height: 100%;
            background: rgba(255, 255, 255, 0.1);
            cursor: ew-resize;
            z-index: 2147483648;
            transition: background 0.2s;
        `;

        // Resize logic
        let isResizing = false;
        let startX = 0;
        let startWidth = 0;

        resizeHandle.addEventListener('mouseenter', () => {
            resizeHandle.style.background = '#ff6b00';
        });

        resizeHandle.addEventListener('mouseleave', () => {
            if (!isResizing) {
                resizeHandle.style.background = 'rgba(255, 255, 255, 0.1)';
            }
        });

        resizeHandle.addEventListener('mousedown', (e) => {
            isResizing = true;
            startX = e.clientX;
            startWidth = parseInt(sidebarIframe.style.width);
            document.body.style.cursor = 'ew-resize';
            document.body.style.userSelect = 'none';
            e.preventDefault();
        });

        document.addEventListener('mousemove', (e) => {
            if (!isResizing) return;
            const delta = startX - e.clientX;
            const newWidth = Math.min(Math.max(startWidth + delta, 300), 800);
            sidebarIframe.style.width = newWidth + 'px';
            resizeHandle.style.right = newWidth + 'px';
        });

        document.addEventListener('mouseup', () => {
            if (isResizing) {
                isResizing = false;
                document.body.style.cursor = '';
                document.body.style.userSelect = '';
                resizeHandle.style.background = 'rgba(255, 255, 255, 0.1)';
                // Save width
                chrome.storage.local.set({ 'revizia_sidebar_width': sidebarIframe.style.width });
            }
        });

        // Restore saved width
        chrome.storage.local.get(['revizia_sidebar_width'], (result) => {
            if (result.revizia_sidebar_width) {
                sidebarIframe.style.width = result.revizia_sidebar_width;
                resizeHandle.style.right = result.revizia_sidebar_width;
                document.body.style.width = 'calc(100% - ' + result.revizia_sidebar_width + ')';
                document.body.style.marginRight = result.revizia_sidebar_width;
            }
        });

        document.body.appendChild(resizeHandle);
        document.body.appendChild(sidebarIframe);
    }

    document.body.style.transition = 'width 0.3s ease, margin-right 0.3s ease';
    document.body.style.width = 'calc(100% - ' + sidebarIframe.style.width + ')';
    document.body.style.marginRight = sidebarIframe.style.width;
    sidebarIframe.style.display = 'block';
    sidebarOpen = true;
}

function closeSidebar() {
    document.body.style.width = '';
    document.body.style.marginRight = '';
    if (sidebarIframe) {
        sidebarIframe.style.display = 'none';
    }
    sidebarOpen = false;
}

// Listen for messages from the sidebar (Legacy/Direct)
window.addEventListener('message', (event) => {
    // Log all messages for debugging
    if (event.data && event.data.type) {
        console.log('RevExt1 Content: Received message', event.data.type, 'from origin:', event.origin);
    }

    // Validate message origin is from our extension
    const extensionOrigin = chrome.runtime.getURL('').slice(0, -1); // Remove trailing slash
    if (!event.origin.startsWith('chrome-extension://') && !event.origin.startsWith('moz-extension://')) {
        return; // Ignore messages not from extension
    }

    if (event.data.type === 'GET_DOM_INFO') {
        console.log('RevExt1 Content: Processing GET_DOM_INFO');
        const pageInfo = {
            title: document.title,
            url: window.location.href,
            bodyText: document.body.innerText.substring(0, 5000)
        };
        if (sidebarIframe && sidebarIframe.contentWindow) {
            try {
                sidebarIframe.contentWindow.postMessage({
                    type: 'DOM_INFO_RESPONSE',
                    payload: pageInfo
                }, extensionOrigin);
                console.log('RevExt1 Content: Sent DOM_INFO_RESPONSE');
            } catch (err) {
                console.error('RevExt1: postMessage failed:', err);
            }
        }
    } else if (event.data.type === 'ALTER_DOM' || event.data.type === 'INTERACT_DOM') {
        console.log('RevExt1 Content: Processing DOM interaction', event.data);
        showDebugToast(`Ação: ${event.data.action} | Seletor: ${event.data.selector || event.data.color || 'N/A'}`);
        handleDOMInteraction(event.data);
    }
});

// Robust element finder with multiple strategies
function findElement(selector, action) {
    console.log(`RevExt1: findElement called with selector: "${selector}", action: "${action}"`);

    // Strategy 1: Direct CSS selector
    try {
        const directMatch = document.querySelector(selector);
        if (directMatch) {
            console.log('RevExt1: Found via direct CSS selector');
            return directMatch;
        }
    } catch (e) {
        console.log('RevExt1: Invalid CSS selector, trying other strategies');
    }

    // Strategy 2: Find by ID (selector might be just the ID value)
    const byId = document.getElementById(selector) || document.getElementById(selector.replace('#', ''));
    if (byId) {
        console.log('RevExt1: Found via ID');
        return byId;
    }

    // Strategy 3: Find by text content (common for buttons/links)
    const cleanSelector = selector.replace(/^[.#]/, '').toLowerCase().trim();

    // Try buttons first
    const buttons = document.querySelectorAll('button, [role="button"], input[type="button"], input[type="submit"]');
    for (const btn of buttons) {
        const btnText = (btn.textContent || btn.value || '').toLowerCase().trim();
        if (btnText === cleanSelector || btnText.includes(cleanSelector)) {
            console.log('RevExt1: Found button via text content');
            return btn;
        }
    }

    // Try links
    const links = document.querySelectorAll('a');
    for (const link of links) {
        const linkText = (link.textContent || '').toLowerCase().trim();
        if (linkText === cleanSelector || linkText.includes(cleanSelector)) {
            console.log('RevExt1: Found link via text content');
            return link;
        }
    }

    // Strategy 4: Find by aria-label or title
    const byAria = document.querySelector(`[aria-label*="${cleanSelector}" i], [title*="${cleanSelector}" i]`);
    if (byAria) {
        console.log('RevExt1: Found via aria-label/title');
        return byAria;
    }

    // Strategy 5: Find by class name (partial match)
    const byClass = document.querySelector(`[class*="${cleanSelector}" i]`);
    if (byClass) {
        console.log('RevExt1: Found via partial class match');
        return byClass;
    }

    // Strategy 6: Find input by placeholder
    if (action === 'type') {
        const inputs = document.querySelectorAll('input, textarea');
        for (const input of inputs) {
            const placeholder = (input.placeholder || '').toLowerCase();
            const name = (input.name || '').toLowerCase();
            const ariaLabel = (input.getAttribute('aria-label') || '').toLowerCase();
            if (placeholder.includes(cleanSelector) || name.includes(cleanSelector) || ariaLabel.includes(cleanSelector)) {
                console.log('RevExt1: Found input via placeholder/name/aria-label');
                return input;
            }
        }
    }

    // Strategy 7: Find any clickable element by text
    if (action === 'click') {
        const clickables = document.querySelectorAll('button, a, [onclick], [role="button"], input[type="submit"], input[type="button"]');
        for (const el of clickables) {
            const text = (el.textContent || el.value || '').toLowerCase().trim();
            // Check for partial match
            if (text.includes(cleanSelector) || cleanSelector.includes(text)) {
                console.log('RevExt1: Found clickable via partial text match');
                return el;
            }
        }
    }

    // Strategy 8: Search all elements for data attributes
    const byData = document.querySelector(`[data-testid*="${cleanSelector}" i], [data-id*="${cleanSelector}" i], [name*="${cleanSelector}" i]`);
    if (byData) {
        console.log('RevExt1: Found via data attribute');
        return byData;
    }

    console.log('RevExt1: Element not found with any strategy');
    return null;
}

function handleDOMInteraction(data) {
    console.log('=== RevExt1 Content: handleDOMInteraction ===');
    console.log('Data received:', JSON.stringify(data));

    if (data.type === 'ALTER_DOM') {
        if (data.action === 'change_bg') {
            console.log('RevExt1: Changing background color to', data.color);
            document.body.style.backgroundColor = data.color;
            sendFeedbackToSidebar('success', `Background alterado para ${data.color}`);
        }
    } else if (data.type === 'INTERACT_DOM') {
        console.log('RevExt1 Content: Processing Action', data);
        const { action, selector, text } = data;

        console.log(`RevExt1: Looking for element with selector: "${selector}"`);

        let element = findElement(selector, action);

        if (element) {
            console.log('RevExt1 Content: ✓ Element found', element.tagName, element.className);
            element.scrollIntoView({ behavior: 'smooth', block: 'center' });

            // Highlight element briefly
            const originalTransition = element.style.transition;
            const originalOutline = element.style.outline;
            const originalBoxShadow = element.style.boxShadow;
            element.style.transition = 'all 0.3s';
            element.style.outline = '3px solid #ff6b00';
            element.style.boxShadow = '0 0 10px rgba(255, 107, 0, 0.5)';

            setTimeout(() => {
                element.style.outline = originalOutline;
                element.style.boxShadow = originalBoxShadow;
                element.style.transition = originalTransition;
            }, 1500);

            if (action === 'click') {
                console.log('RevExt1: Executing CLICK');
                try {
                    element.focus();
                    element.click();

                    // Also dispatch mouse events for better compatibility
                    element.dispatchEvent(new MouseEvent('click', {
                        bubbles: true,
                        cancelable: true,
                        view: window
                    }));

                    console.log('RevExt1: ✓ Click executed successfully');
                    sendFeedbackToSidebar('success', `Clique executado em ${element.tagName}`);
                } catch (e) {
                    console.error('RevExt1: Click failed:', e);
                    sendFeedbackToSidebar('error', `Erro ao clicar: ${e.message}`);
                }
            } else if (action === 'type') {
                console.log('RevExt1: Executing TYPE with text:', text);
                element.focus();
                const setNativeValue = (el, value) => {
                    const valueSetter = Object.getOwnPropertyDescriptor(el, 'value')?.set;
                    const prototype = Object.getPrototypeOf(el);
                    const prototypeValueSetter = Object.getOwnPropertyDescriptor(prototype, 'value')?.set;

                    if (valueSetter && valueSetter !== prototypeValueSetter) {
                        prototypeValueSetter.call(el, value);
                    } else if (valueSetter) {
                        valueSetter.call(el, value);
                    } else {
                        el.value = value;
                    }
                };
                try {
                    setNativeValue(element, text);
                } catch (e) {
                    element.value = text;
                }
                element.dispatchEvent(new Event('input', { bubbles: true }));
                element.dispatchEvent(new Event('change', { bubbles: true }));
                element.dispatchEvent(new KeyboardEvent('keyup', { bubbles: true }));
                element.blur();
                console.log('RevExt1: ✓ Type executed successfully');
                sendFeedbackToSidebar('success', `Texto digitado em ${element.tagName}`);
            }
        } else {
            console.error(`RevExt1: ✗ Element NOT found for selector: "${selector}"`);
            sendFeedbackToSidebar('error', `Elemento não encontrado: ${selector}`);
        }
    }
    console.log('=== RevExt1 Content: END handleDOMInteraction ===');
}

function sendFeedbackToSidebar(status, message) {
    if (sidebarIframe && sidebarIframe.contentWindow) {
        try {
            const extensionOrigin = chrome.runtime.getURL('').slice(0, -1);
            sidebarIframe.contentWindow.postMessage({
                type: 'ACTION_FEEDBACK',
                status: status,
                message: message
            }, extensionOrigin);
        } catch (e) {
            console.log('Could not send feedback to sidebar:', e);
        }
    }
}

// Debug toast on the page itself
function showDebugToast(message) {
    const toast = document.createElement('div');
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        left: 20px;
        background: #ff6b00;
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        z-index: 2147483647;
        font-family: sans-serif;
        font-size: 14px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        animation: slideIn 0.3s ease;
    `;
    toast.textContent = `RevExt1: ${message}`;
    document.body.appendChild(toast);
    setTimeout(() => {
        toast.style.opacity = '0';
        toast.style.transition = 'opacity 0.3s';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}
