chrome.action.onClicked.addListener(async (tab) => {
  if (!tab.id) return;

  // Prevent running on restricted URLs
  if (tab.url.startsWith("chrome://") || tab.url.startsWith("edge://") || tab.url.startsWith("about:")) {
    console.warn("RevExt1: Cannot run on this page.");
    return;
  }

  try {
    // Try to send message
    await chrome.tabs.sendMessage(tab.id, { action: "toggleSidebar" });
  } catch (err) {
    console.log("RevExt1: Content script not ready, injecting...", err);
    // If failed, inject script
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ["content.js"]
      });
      // Retry message
      setTimeout(() => {
        chrome.tabs.sendMessage(tab.id, { action: "toggleSidebar" });
      }, 100);
    } catch (injectionErr) {
      console.error("RevExt1: Failed to inject script", injectionErr);
    }
  }
});

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "CAPTURE_SCREENSHOT") {
    chrome.tabs.captureVisibleTab(null, { format: "png" }, (dataUrl) => {
      if (chrome.runtime.lastError) {
        sendResponse({ error: chrome.runtime.lastError.message });
      } else {
        sendResponse({ dataUrl: dataUrl });
      }
    });
    return true; // Indicates async response
  }

  if (request.action === "FETCH_URL") {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 10000); // 10s timeout

    fetch(request.url, { signal: controller.signal })
      .then(response => {
        clearTimeout(timeoutId);
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        return response.text();
      })
      .then(text => {
        sendResponse({ success: true, content: text });
      })
      .catch(error => {
        clearTimeout(timeoutId);
        console.error('[Background] Fetch error:', error);
        sendResponse({ success: false, error: error.message });
      });
    return true;
  }

  // Deep Scan Handler - Now handled in background to avoid CSP issues
  if (request.action === "START_DEEP_SCAN") {
    (async () => {
      try {
        const results = await crawlPages(request.url, 2, 15);
        sendResponse({ success: true, data: results });
      } catch (err) {
        console.error('[Background] Deep scan error:', err);
        sendResponse({ success: false, error: err.message });
      }
    })();
    return true;
  }
});

// Crawl pages function - runs in background to avoid CSP issues
async function crawlPages(startUrl, maxDepth = 2, maxPages = 15) {
  console.log(`[Background] Starting crawl for ${startUrl} (Depth: ${maxDepth}, MaxPages: ${maxPages})`);

  const visited = new Set();
  const queue = [{ url: startUrl, depth: 0 }];
  const results = [];
  const startDomain = new URL(startUrl).hostname;

  while (queue.length > 0 && results.length < maxPages) {
    const { url, depth } = queue.shift();

    if (visited.has(url)) continue;
    visited.add(url);

    try {
      console.log(`[Background] Fetching ${url} (Depth: ${depth})`);

      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 8000); // 8s timeout per page

      const response = await fetch(url, { signal: controller.signal });
      clearTimeout(timeoutId);

      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('text/html')) {
        console.log(`[Background] Skipping non-HTML content: ${url}`);
        continue;
      }

      const html = await response.text();

      // Send to offscreen for parsing (DOM operations only, no fetch)
      await setupOffscreenDocument('offscreen.html');
      const parseResult = await new Promise((resolve) => {
        chrome.runtime.sendMessage({
          target: "OFFSCREEN",
          type: "PARSE_HTML",
          data: { html, url, depth }
        }, (response) => {
          resolve(response);
        });
      });

      if (parseResult && parseResult.success) {
        results.push(parseResult.data);

        // Extract links for next depth level
        if (depth < maxDepth && results.length < maxPages && parseResult.data.links) {
          for (const link of parseResult.data.links) {
            try {
              const nextUrl = new URL(link, url).href;
              if (nextUrl.startsWith('http') && new URL(nextUrl).hostname.includes(startDomain) && !visited.has(nextUrl)) {
                queue.push({ url: nextUrl, depth: depth + 1 });
              }
            } catch (e) {
              // Invalid URL
            }
          }
        }
      }

    } catch (error) {
      console.error(`[Background] Error crawling ${url}:`, error.message);
      results.push({ url, error: error.message, depth });
    }
  }

  console.log(`[Background] Crawl complete. Processed ${results.length} pages.`);
  return results;
}

// High-Performance Offscreen Lifecycle Management
async function setupOffscreenDocument(path) {
  // Check if offscreen document already exists
  if (await chrome.offscreen.hasDocument()) {
    return;
  }

  // Create offscreen document
  await chrome.offscreen.createDocument({
    url: path,
    reasons: ['WORKERS', 'DOM_PARSER'],
    justification: 'Running heavy text analysis and WASM modules background tasks'
  });
}

// Initialize Offscreen Core
setupOffscreenDocument('offscreen.html');

// Ensure offscreen document is active when messages are sent
chrome.runtime.onMessage.addListener(async (msg) => {
  if (msg.target === 'OFFSCREEN') {
    await setupOffscreenDocument('offscreen.html');
  }
});
